<?php
/////////////////////////////////////////////////////////////////
/// getID3() by James Heinrich <info@getid3.org>               //
//  available at http://getid3.sourceforge.net                 //
//            or http://www.getid3.org                         //
/////////////////////////////////////////////////////////////////
// See readme.txt for more details                             //
/////////////////////////////////////////////////////////////////
//                                                             //
// module.archive.tiff.php                                     //
// module for analyzing TIFF files                             //
// dependencies: NONE                                          //
//                                                            ///
/////////////////////////////////////////////////////////////////


class getid3_tiff
{

	function getid3_tiff(&$fd, &$ThisFileInfo) {

		fseek($fd, $ThisFileInfo['avdataoffset'], SEEK_SET);
		$TIFFheader = fread($fd, 4);

		switch (substr($TIFFheader, 0, 2)) {
			case 'II':
				$ThisFileInfo['tiff']['byte_order'] = 'Intel';
				break;
			case 'MM':
				$ThisFileInfo['tiff']['byte_order'] = 'Motorola';
				break;
			default:
				$ThisFileInfo['error'][] = 'Invalid TIFF byte order identifier ('.substr($TIFFheader, 0, 2).') at offset '.$ThisFileInfo['avdataoffset'];
				return false;
				break;
		}

		$ThisFileInfo['fileformat']          = 'tiff';
		$ThisFileInfo['video']['dataformat'] = 'tiff';
		$ThisFileInfo['video']['lossless']   = true;
		$ThisFileInfo['tiff']['ifd']         = array();
		$CurrentIFD                          = array();

		$FieldTypeByteLength = array(1=>1, 2=>1, 3=>2, 4=>4, 5=>8);

		$nextIFDoffset = $this->TIFFendian2Int(fread($fd, 4), $ThisFileInfo['tiff']['byte_order']);

		while ($nextIFDoffset > 0) {

			$CurrentIFD['offset'] = $nextIFDoffset;

			fseek($fd, $ThisFileInfo['avdataoffset'] + $nextIFDoffset, SEEK_SET);
			$CurrentIFD['fieldcount'] = $this->TIFFendian2Int(fread($fd, 2), $ThisFileInfo['tiff']['byte_order']);

			for ($i = 0; $i < $CurrentIFD['fieldcount']; $i++) {
				$CurrentIFD['fields'][$i]['raw']['tag']    = $this->TIFFendian2Int(fread($fd, 2), $ThisFileInfo['tiff']['byte_order']);
				$CurrentIFD['fields'][$i]['raw']['type']   = $this->TIFFendian2Int(fread($fd, 2), $ThisFileInfo['tiff']['byte_order']);
				$CurrentIFD['fields'][$i]['raw']['length'] = $this->TIFFendian2Int(fread($fd, 4), $ThisFileInfo['tiff']['byte_order']);
				$CurrentIFD['fields'][$i]['raw']['offset'] =                       fread($fd, 4);

				switch ($CurrentIFD['fields'][$i]['raw']['type']) {
					case 1: // BYTE  An 8-bit unsigned integer.
						if ($CurrentIFD['fields'][$i]['raw']['length'] <= 4) {
							$CurrentIFD['fields'][$i]['value']  = $this->TIFFendian2Int(substr($CurrentIFD['fields'][$i]['raw']['offset'], 0, 1), $ThisFileInfo['tiff']['byte_order']);
						} else {
							$CurrentIFD['fields'][$i]['offset'] = $this->TIFFendian2Int($CurrentIFD['fields'][$i]['raw']['offset'], $ThisFileInfo['tiff']['byte_order']);
						}
						break;

					case 2: // ASCII 8-bit bytes  that store ASCII codes; the last byte must be null.
						if ($CurrentIFD['fields'][$i]['raw']['length'] <= 4) {
							$CurrentIFD['fields'][$i]['value']  = substr($CurrentIFD['fields'][$i]['raw']['offset'], 3);
						} else {
							$CurrentIFD['fields'][$i]['offset'] = $this->TIFFendian2Int($CurrentIFD['fields'][$i]['raw']['offset'], $ThisFileInfo['tiff']['byte_order']);
						}
						break;

					case 3: // SHORT A 16-bit (2-byte) unsigned integer.
						if ($CurrentIFD['fields'][$i]['raw']['length'] <= 2) {
							$CurrentIFD['fields'][$i]['value']  = $this->TIFFendian2Int(substr($CurrentIFD['fields'][$i]['raw']['offset'], 0, 2), $ThisFileInfo['tiff']['byte_order']);
						} else {
							$CurrentIFD['fields'][$i]['offset'] = $this->TIFFendian2Int($CurrentIFD['fields'][$i]['raw']['offset'], $ThisFileInfo['tiff']['byte_order']);
						}
						break;

					case 4: // LONG  A 32-bit (4-byte) unsigned integer.
						if ($CurrentIFD['fields'][$i]['raw']['length'] <= 1) {
							$CurrentIFD['fields'][$i]['value']  = $this->TIFFendian2Int($CurrentIFD['fields'][$i]['raw']['offset'], $ThisFileInfo['tiff']['byte_order']);
						} else {
							$CurrentIFD['fields'][$i]['offset'] = $this->TIFFendian2Int($CurrentIFD['fields'][$i]['raw']['offset'], $ThisFileInfo['tiff']['byte_order']);
						}
						break;

					case 5: // RATIONAL   Two LONGs:  the first represents the numerator of a fraction, the second the denominator.
						$CurrentIFD['fields'][$i]['offset'] = $this->TIFFendian2Int($CurrentIFD['fields'][$i]['raw']['offset'], $ThisFileInfo['tiff']['byte_order']);
						break;
				}
			}

			$ThisFileInfo['tiff']['ifd'][] = $CurrentIFD;
			$CurrentIFD = array();
			$nextIFDoffset = $this->TIFFendian2Int(fread($fd, 4), $ThisFileInfo['tiff']['byte_order']);

		}

		foreach ($ThisFileInfo['tiff']['ifd'] as $IFDid => $IFDarray) {
			foreach ($IFDarray['fields'] as $key => $fieldarray) {
				if (isset($fieldarray['value'])) {
					$ThisFileInfo['tiff']['ifd'][$IFDid]['fields'][$key]['raw']['data'] = $fieldarray['value'];
				} else {
					$len = $fieldarray['raw']['length'] * $FieldTypeByteLength[$fieldarray['raw']['type']];
					if ($len) {
						fseek($fd, $fieldarray['offset'], SEEK_SET);
						$ThisFileInfo['tiff']['ifd'][$IFDid]['fields'][$key]['raw']['data'] = fread($fd, $len);
					}
				}

				if (isset($fieldarray['value'])) {
					$tag_value = (array)$fieldarray['value'];
				} else if ($fieldarray['raw']['type'] == 2) {
					$tag_value = (array)$ThisFileInfo['tiff']['ifd'][$IFDid]['fields'][$key]['raw']['data'];
				} else {
					$tag_value = array();
					for ($i = 0; $i < $fieldarray['raw']['length']; $i++) {
						$tag_value[] = $this->TIFFendian2Num(
							substr(
								$ThisFileInfo['tiff']['ifd'][$IFDid]['fields'][$key]['raw']['data'],
								$i * $FieldTypeByteLength[$fieldarray['raw']['type']],
								$FieldTypeByteLength[$fieldarray['raw']['type']]
							), $fieldarray['raw']['type'], $ThisFileInfo['tiff']['byte_order']
						);
					}
				}
				if ( count($tag_value) == 1 ) $tag_value = $tag_value[0];
				$ThisFileInfo['tiff']['ifd'][$IFDid]['tags'][$this->TIFFtagName($fieldarray['raw']['tag'])] = $tag_value;


				switch ($fieldarray['raw']['tag']) {
					case 254: // NewSubfileType
						$ThisFileInfo['tiff']['ifd'][$IFDid]['tags']['SubfileType_text'] = $this->TIFFsubfileType( $fieldarray['value'], 0 );
						break;
					case 255: // SubfileType
						if(empty( $ThisFileInfo['tiff']['ifd'][$IFDid]['tags']['SubfileType_text'] ))
							$ThisFileInfo['tiff']['ifd'][$IFDid]['tags']['SubfileType_text'] = $this->TIFFsubfileType( 0,$fieldarray['value'] );
						break;
					case 259: // Compression
						$ThisFileInfo['tiff']['ifd'][$IFDid]['tags']['Compression_text'] = $this->TIFFcompressionMethod($fieldarray['value']);
						//$ThisFileInfo['video']['codec']
						break;

					case 262: // PhotometricInterpretation
						$ThisFileInfo['tiff']['ifd'][$IFDid]['tags']['PhotometricInterpretation_text'] = $this->TIFFphotometricInterpretation($fieldarray['value']);
						break;

					case 296: // ResolutionUnit
						$ThisFileInfo['tiff']['ifd'][$IFDid]['tags']['ResolutionUnit_text'] = $this->TIFFresolutionUnit($fieldarray['value']);
						break;

					case 338: // ExtraSamples
						$ThisFileInfo['tiff']['ifd'][$IFDid]['tags']['PhotometricInterpretation_text'] .= ' + alpha';
						break;

					default:
						break;
				}

				if ($fieldarray['raw']['type'] == 2) {
					@$ThisFileInfo['tiff']['comments'][$this->TIFFtagName($fieldarray['raw']['tag'])][] = substr( $ThisFileInfo['tiff']['ifd'][$IFDid]['fields'][$key]['raw']['data'], 0, -1 );
				}
			}
		}

		$CurrentIFD =& reset($ThisFileInfo['tiff']['ifd']);
		$ThisFileInfo['video']['resolution_x'] = $CurrentIFD['tags']['ImageWidth'];
		$ThisFileInfo['video']['resolution_y'] = $CurrentIFD['tags']['ImageLength'];
		$ThisFileInfo['video']['bits_per_sample'] = array_sum( (array)$CurrentIFD['tags']['BitsPerSample'] );
		$ThisFileInfo['video']['codec'] = $CurrentIFD['tags']['Compression_text'];

		return true;
	}


	function TIFFendian2Int($bytestring, $byteorder) {
		if ($byteorder == 'Intel') {
			return getid3_lib::LittleEndian2Int($bytestring);
		} elseif ($byteorder == 'Motorola') {
			return getid3_lib::BigEndian2Int($bytestring);
		}
		return false;
	}

	function TIFFendian2Num($bytestring, $type, $byteorder) {
		switch ($type) {
			case 1: // BYTE
			case 3: // SHORT
			case 4: // LONG
				return $this->TIFFendian2Int($bytestring, $byteorder);

			case 5: // RATIONAL
				$numerator = $this->TIFFendian2Int(substr($bytestring, 0, 4), $byteorder);
				$denominator = $this->TIFFendian2Int(substr($bytestring, 4, 4), $byteorder);
				if ($denominator > 0) return $numerator / $denominator;
				return false;

			default:
				return false;
		}
	}

	function TIFFsubfileType($new,$old) {
		$a = array();
		if ( ( $new & 0x01 ) || ($old==2) ) $a[] = 'reduced-resolution';
		if ( ( $new & 0x02 ) || ($old==3) ) $a[] = 'part of multi-page';
		if ( $new & 0x04 ) $a[] = 'transparency mask';
		if ( $new & 0x10 ) $a[] = 'mixed raster content';
		return empty($a) ? 'full-resolution' : implode(', ',$a);
	}

	function TIFFcompressionMethod($id) {
		static $TIFFcompressionMethod = array();
		if (empty($TIFFcompressionMethod)) {
			$TIFFcompressionMethod = array(
				1     => 'Uncompressed',
				2     => 'Huffman',
				3     => 'CCITT Group 3 Fax',
				4     => 'CCITT Group 4 Fax',
				5     => 'LZW',
				6     => 'JPEG',
				32773 => 'PackBits',
			);
		}
		return (isset($TIFFcompressionMethod[$id]) ? $TIFFcompressionMethod[$id] : "unknown/invalid ($id)");
	}

	function TIFFphotometricInterpretation($id) {
		static $TIFFphotometricInterpretation = array();
		if (empty($TIFFphotometricInterpretation)) {
			$TIFFphotometricInterpretation = array(
				0     => 'Grayscale (white is zero)',
				1     => 'Grayscale (black is zero)',
				2     => 'RGB',
				3     => 'Palette',
				4     => 'Transparency mask',
				5     => 'CMYK',
				6     => 'YCbCr',
				8     => 'CIELab',
			);
		}
		return (isset($TIFFphotometricInterpretation[$id]) ? $TIFFphotometricInterpretation[$id] : "unknown/invalid ($id)");
	}

	function TIFFresolutionUnit($id) {
		static $TIFFresolutionUnit = array();
		if (empty($TIFFresolutionUnit)) {
			$TIFFresolutionUnit = array(
				1     => '',
				2     => 'inch',
				3     => 'cm',
			);
		}
		return (isset($TIFFresolutionUnit[$id]) ? $TIFFresolutionUnit[$id] : "unknown/invalid ($id)");
	}

	function TIFFtagName($id) {
		static $TIFFtagName = array();
		if (empty($TIFFtagName)) {
			$TIFFtagName = array(
				254 => 'NewSubfileType',
				255 => 'SubfileType',
				256 => 'ImageWidth',
				257 => 'ImageLength',
				258 => 'BitsPerSample',
				259 => 'Compression',
				262 => 'PhotometricInterpretation',
				263 => 'Threshholding',
				264 => 'CellWidth',
				265 => 'CellLength',
				266 => 'FillOrder',
				269 => 'DocumentName',
				270 => 'ImageDescription',
				271 => 'Make',
				272 => 'Model',
				273 => 'StripOffsets',
				274 => 'Orientation',
				277 => 'SamplesPerPixel',
				278 => 'RowsPerStrip',
				279 => 'StripByteCounts',
				280 => 'MinSampleValue',
				281 => 'MaxSampleValue',
				282 => 'XResolution',
				283 => 'YResolution',
				284 => 'PlanarConfiguration',
				285 => 'PageName',
				286 => 'XPosition',
				287 => 'YPosition',
				288 => 'FreeOffsets',
				289 => 'FreeByteCounts',
				290 => 'GrayResponseUnit',
				291 => 'GrayResponseCurve',
				292 => 'T4Options',
				293 => 'T6Options',
				296 => 'ResolutionUnit',
				297 => 'PageNumber',
				301 => 'TransferFunction',
				305 => 'Software',
				306 => 'DateTime',
				315 => 'Artist',
				316 => 'HostComputer',
				317 => 'Predictor',
				318 => 'WhitePoint',
				319 => 'PrimaryChromaticities',
				320 => 'ColorMap',
				321 => 'HalftoneHints',
				322 => 'TileWidth',
				323 => 'TileLength',
				324 => 'TileOffsets',
				325 => 'TileByteCounts',
				332 => 'InkSet',
				333 => 'InkNames',
				334 => 'NumberOfInks',
				336 => 'DotRange',
				337 => 'TargetPrinter',
				338 => 'ExtraSamples',
				339 => 'SampleFormat',
				340 => 'SMinSampleValue',
				341 => 'SMaxSampleValue',
				342 => 'TransferRange',
				512 => 'JPEGProc',
				513 => 'JPEGInterchangeFormat',
				514 => 'JPEGInterchangeFormatLngth',
				515 => 'JPEGRestartInterval',
				517 => 'JPEGLosslessPredictors',
				518 => 'JPEGPointTransforms',
				519 => 'JPEGQTables',
				520 => 'JPEGDCTables',
				521 => 'JPEGACTables',
				529 => 'YCbCrCoefficients',
				530 => 'YCbCrSubSampling',
				531 => 'YCbCrPositioning',
				532 => 'ReferenceBlackWhite',
				33432 => 'Copyright',
			);
		}
		return (isset($TIFFtagName[$id]) ? $TIFFtagName[$id] : "unknown/invalid ($id)");
	}

}


?>
